﻿using FellowOakDicom;
using FellowOakDicom.Network;
using FellowOakDicom.Network.Client;
using FellowOakDicom.Network.Client.EventArguments;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace DicomCmove
{
    public class Program
    {
        // Dr. Harvey graciously provides a free DICOM server to connect and play with
        private static string patientName = "Anon*";
        private static string seriesDescription = "KNEE LEFT*";
        private static string QRServerHost = "192.168.88.26";
        private static int QRServerPort = 11112;
        private static string QRServerAET = "XRAYARCHIVE";
        private static string localAET = "LOCALAET";
        private static string moveAET = "MOVEAET";
        private static List<string> studyUIDs = new List<string>();
        private static List<string> seriesUIDs = new List<string>();

        static async Task Main(string[] args)
        {
            try
            {
                if (await GetDicomStudyUidsAsync(patientName))
                {
                    foreach (string studyUID in studyUIDs) 
                    {
                        if (await GetDicomSeriesUidsAsync(studyUID))
                        {
                            foreach (string seriesUID in seriesUIDs)
                            {
                                MoveSeriesUidsAsync(studyUID, seriesUID);
                            }                            
                        }
                        else
                        {
                            LogToDebugConsole("No series found");
                        }
                    }
                }
                else 
                {
                    LogToDebugConsole("No studies found");
                }
                Console.ReadLine();
            }
            catch (Exception e)
            {
                //In real life, do something about this exception
                LogToDebugConsole($"Error request -> {e.StackTrace}");
            }
        }

        private static async Task<bool> GetDicomStudyUidsAsync(string patientNameForSearch)
        {
            try
            {
                // Создаем клиент для установления ассоциации
                var dicomClient = DicomClientFactory.Create(QRServerHost, QRServerPort, false, localAET, QRServerAET);
                dicomClient.NegotiateAsyncOps();

                //Добавляем обработчик, который будет получать уведомления о любых отклонениях ассоциации.
                dicomClient.AssociationRejected += ClientOnAssociationRejected;

                //Добавляем обработчик для получения уведомлений о любой информации об ассоциации при успешных соединениях.
                dicomClient.AssociationAccepted += ClientOnAssociationAccepted;

                //Добавляем обработчик для уведомления об успешном освобождении ассоциации — это также может быть вызвано удаленным узлом.
                dicomClient.AssociationReleased += ClientOnAssociationReleased;
                
                // Создание нового экземпляра запроса уровня "Study Root"
                var studyRequest = new DicomCFindRequest(DicomQueryRetrieveLevel.Study);
                studyRequest.Dataset.AddOrUpdate(DicomTag.SpecificCharacterSet, "ISO_IR 192");

                // Фильтр по имени пациента 
                studyRequest.Dataset.AddOrUpdate(DicomTag.PatientName, patientNameForSearch);
                // Запрос на возврат тега StudyInstanceUID
                studyRequest.Dataset.AddOrUpdate(DicomTag.StudyInstanceUID, string.Empty);

                // Найденные идентификаторы исследований добавляем в список
                studyRequest.OnResponseReceived += (req, response) =>
                {                    
                    if (response.Status.ToString() != "Success")
                    {// Пока сервер не вернет "Success" добавляем в список найденные идентификаторы исследований
                        studyUIDs.Add(response.Dataset.GetSingleValueOrDefault<string>(DicomTag.StudyInstanceUID, ""));
                        LogToDebugConsole($"Found study: {response.Dataset.GetSingleValueOrDefault<string>(DicomTag.StudyInstanceUID, "")}");
                    }
                };

                // Добавляем запрос на поиск исследований пациента 
                await dicomClient.AddRequestAsync(studyRequest);
                await dicomClient.SendAsync(); // Собственно посылка запроса
                                
                if (studyUIDs.Count > 0) return true;
                else return false;
            }
            catch (Exception e)
            {   
                LogToDebugConsole($"Error get study list: {e.StackTrace}");
                return false;
            }
        }

        private static async Task<bool> GetDicomSeriesUidsAsync(string studyUIDForSearch)
        {
            try
            {
                // Создаем клиент для установления ассоциации
                var dicomClient = DicomClientFactory.Create(QRServerHost, QRServerPort, false, localAET, QRServerAET);
                dicomClient.NegotiateAsyncOps();                
                dicomClient.AssociationRejected += ClientOnAssociationRejected;
                dicomClient.AssociationAccepted += ClientOnAssociationAccepted;
                dicomClient.AssociationReleased += ClientOnAssociationReleased;

                // Создание нового экземпляра запроса уровня "Series Root"
                var seriesRequest = new DicomCFindRequest(DicomQueryRetrieveLevel.Series);
                seriesRequest.Dataset.AddOrUpdate(DicomTag.SpecificCharacterSet, "ISO_IR 192");

                // Фильтр по UID исследования и описанию серии
                seriesRequest.Dataset.AddOrUpdate(DicomTag.StudyInstanceUID, studyUIDForSearch);
                seriesRequest.Dataset.AddOrUpdate(DicomTag.SeriesDescription, seriesDescription);
                // Запрос на возврат тега SeriesInstanceUID
                seriesRequest.Dataset.AddOrUpdate(DicomTag.SeriesInstanceUID, string.Empty);

                // Найденные идентификаторы исследований добавляем в список
                seriesRequest.OnResponseReceived += (req, response) =>
                {
                    if (response.Status.ToString() != "Success")
                    {
                        seriesUIDs.Add(response.Dataset.GetSingleValueOrDefault<string>(DicomTag.SeriesInstanceUID, ""));
                        LogToDebugConsole($"Found series: {response.Dataset.GetSingleValueOrDefault<string>(DicomTag.SeriesInstanceUID, "")}");
                    }
                };

                // Добавляем запрос на поиск ctcthbqhbq пациента 
                await dicomClient.AddRequestAsync(seriesRequest);
                await dicomClient.SendAsync(); // Собственно посылка запроса с ожиданием результатов

                if (seriesUIDs.Count > 0) return true;
                else return false;
            }
            catch (Exception e)
            {
                LogToDebugConsole($"Error get series list: {e.StackTrace}");
                return false;
            }
        }

        private static async void MoveSeriesUidsAsync(string studyUid, string seriesUid)
        {
            try
            {
                // Создаем клиент для установления ассоциации
                var dicomClient = DicomClientFactory.Create(QRServerHost, QRServerPort, false, localAET, QRServerAET);
                dicomClient.NegotiateAsyncOps();
                dicomClient.AssociationRejected += ClientOnAssociationRejected;                       
                dicomClient.AssociationAccepted += ClientOnAssociationAccepted;
                dicomClient.AssociationReleased += ClientOnAssociationReleased;

                // Создание запроса C-MOVE
                var cmove = new DicomCMoveRequest(moveAET, studyUid, seriesUid);
                              
                cmove.OnResponseReceived += (DicomCMoveRequest request, DicomCMoveResponse response) =>
                {
                    if (response.Status.ToString() == "Success")
                    {
                        //string studyUID = request.Dataset?.GetSingleValueOrDefault<string>(DicomTag.StudyInstanceUID, "");
                        string seriesUID = request.Dataset?.GetSingleValueOrDefault<string>(DicomTag.SeriesInstanceUID, "");
                        LogToDebugConsole($"Move series SUCCESS  -------------------------------->>>> {seriesUID}");
                    }
                    else
                    {
                        LogToDebugConsole($"Move series {seriesUid} - {response.Status}");
                    }
                };

                // Добавляем запрос на поиск ctcthbqhbq пациента 
                await dicomClient.AddRequestAsync(cmove);               
                await dicomClient.SendAsync(); // Собственно посылка запроса с ожиданием результатов
            }
            catch (Exception e)
            {
                LogToDebugConsole($"Error move series: {e.StackTrace}");
                           }
        }
              
        private static void ClientOnAssociationReleased(object sender, EventArgs e)
        {
            LogToDebugConsole("Assoacition was released");
        }
        private static void ClientOnAssociationRejected(object sender, AssociationRejectedEventArgs e)
        {
            LogToDebugConsole($"Association was rejected. Rejected Reason:{e.Reason}");
        }
        private static void ClientOnAssociationAccepted(object sender, AssociationAcceptedEventArgs e)
        {
            var association = e.Association;
            LogToDebugConsole($"Association was accepted by remote host: {association.RemoteHost} running on port: {association.RemotePort}");

            foreach (var presentationContext in association.PresentationContexts)
            {
                if (presentationContext.Result == DicomPresentationContextResult.Accept)
                {
                    LogToDebugConsole($"\t {presentationContext.AbstractSyntax} was accepted");
                    LogToDebugConsole($"\t Negotiation result was: {presentationContext.GetResultDescription()}");
                    LogToDebugConsole($"\t Abstract syntax accepted: {presentationContext.AbstractSyntax}");
                    LogToDebugConsole($"\t Transfer syntax accepted: {presentationContext.AcceptedTransferSyntax.ToString()}");
                }
                else
                {
                    LogToDebugConsole($"\t Presentation context with proposed abstract syntax of '{presentationContext.AbstractSyntax}' was not accepted");
                    LogToDebugConsole($"\t Reject reason was {presentationContext.GetResultDescription()}");
                }
            }
        }
        private static void LogToDebugConsole(string informationToLog)
        {
            Console.WriteLine(informationToLog);
        }

    }

}
